<?php

namespace App\Utility\SetupWizard;

use Config;
use Artisan;
use DB;

class SetupWizard {
    private $php_version = '8.2';

    public function get_setup_requirements() {
        $requirements = [
            phpversion() < $this->php_version ? 'Minimum PHP Version ' . $this->php_version . ' required' : null,
            !ini_get('file_uploads') ? 'File Uploads feature needs to be enabled!' : null,
            !class_exists('PDO') ? 'MySQL PDO feature needs to be enabled!' : null,
            !extension_loaded('mbstring') ? 'mbstring extension needs to be loaded!' : null,
            !extension_loaded('zip') ? 'ZIP extension needs to be loaded!' : null,
            !extension_loaded('fileinfo') ? 'fileinfo extension needs to be loaded!' : null,
            !is_writable(base_path('storage/app')) ? 'storage/app directory needs to be writable!' : null,
            !is_writable(base_path('storage/framework')) ? 'storage/framework directory needs to be writable!' : null,
            !is_writable(base_path('storage/logs')) ? 'storage/logs directory needs to be writable!' : null,
            !is_writable(base_path('public/images')) ? 'public/images directory needs to be writable!' : null,
            !is_writable(base_path('.env')) ? '.env file needs to be writable!' : null,
        ];

        return array_filter($requirements);
    }

    public function create_db_tables($host, $database, $username, $password) {
        if (!$this->is_db_valid($host, $database, $username, $password)) {
            return false;
        }

        $this->save_db_variables($host, 3306, $database, $username, $password);
        return true;
    }

    public function run_migrations() {
        try {
            @set_time_limit(300);
            Artisan::call('migrate:fresh', ['--force' => true]);
            Artisan::call('db:seed', ['--force' => true]);
            return true;
        } catch (\Throwable $th) {
            return false;
        }
    }

    public function is_db_valid($host, $database, $username, $password) {
        Config::set('database.connections.install_test', [
            'host' => $host,
            'port' => env('DB_PORT', '3306'),
            'database' => $database,
            'username' => $username,
            'password' => $password,
            'driver' => env('DB_CONNECTION', 'mysql'),
            'charset' => env('DB_CHARSET', 'utf8mb4'),
        ]);

        try {
            DB::connection('install_test')->getPdo();
        } catch (\Exception $e) {
            return false;
        }

        DB::purge('install_test');
        return true;
    }

    public function save_db_variables($host, $port, $database, $username, $password) {
        $this->update_env([
            'DB_HOST' => $host,
            'DB_PORT' => $port,
            'DB_DATABASE' => $database,
            'DB_USERNAME' => $username,
            'DB_PASSWORD' => $password,
        ]);

        $con = env('DB_CONNECTION', 'mysql');
        Config::set("database.connections.$con", array_merge(
            Config::get("database.connections.$con"),
            compact('host', 'database', 'username', 'password')
        ));

        DB::purge($con);
        DB::reconnect($con);
    }

    public function finish_setup() {
        $this->update_env([
            'APP_NAME' => "\"Casual Fitness\"",
            'APP_ENV' => 'local',
            'APP_INSTALLED' => 'true',
            'APP_DEBUG' => 'false',
            'APP_URL' => url(''),
        ]);
    }

   	public function update_env($data) {
		if (empty($data) || !is_array($data) || !is_file(base_path('.env'))) {
			return false;
		}

		$env = file_get_contents(base_path('.env'));

		$env = explode("\n", $env);

		foreach ($data as $key => $value) {
			foreach ($env as $env_key => $env_value) {
				$entry = explode('=', $env_value, 2);

				if ($entry[0] == $key) {
					$env[$env_key] = $key . '=' . $value;
				} else {
					$env[$env_key] = $env_value;
				}
			}
		}

		$env = implode("\n", $env);

		file_put_contents(base_path('.env'), $env);

		return true;
	}
}
